/*
 * @(#)Dependency.java  1.0  2006-03-20
 *
 * Copyright (c) 2006 Lucerne University of Applied Sciences and Arts (HSLU)
 * Zentralstrasse 18, Postfach 2858, CH-6002 Lucerne, Switzerland
 * All rights reserved.
 *
 * The copyright of this software is owned by the Lucerne University of Applied 
 * Sciences and Arts (HSLU). You may not use, copy or modify this software, 
 * except in accordance with the license agreement you entered into with HSLU. 
 * For details see accompanying license terms. 
 */

package ch.hslu.cm.pert.model;

import ch.hslu.cm.simulation.*;
import ch.randelshofer.util.*;
import static ch.hslu.cm.pert.model.PertModel.*;
import java.io.IOException;
import org.jhotdraw.util.ResourceBundleUtil;
import org.jhotdraw.xml.DOMInput;
import org.jhotdraw.xml.DOMOutput;
/**
 * Connects elements of a PertModel.
 * <p>
 * <ul>
 * <li>Represents a relationship when connecting an PertTask with a
 * RelationshipSet.</li>
 * <li>Represents a generalization or a specialization when connecting an
 * PertTask with an ISA.</li>
 * <li>Represents an attribute link when connecting an Attribute with an
 * Attribute,  with an PertTask or with a RelationshipSet.</li>
 * </ul>
 *
 *
 *
 * @author Werner Randelshofer
 * @version 1.0 2006-03-20 Created.
 */
public class Dependency extends AbstractRelationship {
    /** Creates a new instance. */
    public Dependency() {
    }
    
    public int getSimulatedConcept() {
        return DEPENDENCY;
    }
    
    public boolean canConnect(SimulatedElement start, SimulatedElement end) {
        int sc = start.getSimulatedConcept();
        int ec = end.getSimulatedConcept();
        
        // Can connect only TASK's
        if (sc != TASK || ec != TASK) {
           // System.out.println("Can't connect, one is not a task "+sc+" "+ec);
            return false;
        }
        
        Task st = (Task) start;
        Task et = (Task) end;
        
        // Disallow multiple connections to same dependent
        if (et.getPredecessors().contains(st)) {
            //System.out.println("Can't connect, "+st+" is predecessor of "+et);
            return false;
        }
        
        // Disallow cyclic connections
        /*
        if (st.isDependentOf(et)) {
            System.out.println("Can't connect. Connecting "+st+" with "+et+" would result in a cycle");
            }*/
        return ! st.isDependentOf(et);
    }
    
    public boolean canConnect(SimulatedElement start) {
        switch (start.getSimulatedConcept()) {
            case PertModel.TASK :
                return true;
            default :
                return false;
        }
    }
    
    public Dependency clone() {
        Dependency that = (Dependency) super.clone();
        return that;
    }
    
    public void read(DOMInput in) throws IOException {
        ResourceBundleUtil labels = PertModel.labels;
        
        name = in.getAttribute("name", labels.getString("relationshipRoleDefaultName"));
        
        in.openElement("start");
        in.openElement("element");
        setStart((SimulatedElement) in.readObject());
        in.closeElement();
        in.closeElement();
        
        in.openElement("end");
        in.openElement("element");
        setEnd((SimulatedElement) in.readObject());
        in.closeElement();
        in.closeElement();
    }
    public void write(DOMOutput out) throws IOException {
        // Write start
        out.openElement("start");
        out.openElement("element");
        out.writeObject(getStart());
        out.closeElement();
        out.closeElement();
        
        // Write end
        out.openElement("end");
        out.openElement("element");
        out.writeObject(getEnd());
        out.closeElement();
        out.closeElement();
    }
    
    /**
     * Handles the disconnection of a connection.
     * Override this method to handle this event.
     */
    protected void handleDisconnected(SimulatedElement start, SimulatedElement end) {
        // Update the end task at all its dependents
        //System.out.println(this+".handleDisconnect");
       ((Task) end).updateStartTime();
    }
    
    /**
     * Handles the connection of a connection.
     * Override this method to handle this event.
     */
    protected void handleConnected(SimulatedElement start, SimulatedElement end) {
        // Update the end task at all its dependents
        //System.out.println(this+".handleConnect");
       ((Task) end).updateStartTime();
    }
}
